package dataFit;

import visad.*;                                                   // The actual interpolation algorithm
import dataFit.InitInterpolation;              				// To solve the constructor problem
import dataFit.NotSameLengthException;      				// Interpolation exception
import java.rmi.RemoteException;                                  // This is used by VisAD

public class Interpolation3DLinear
{
  // Declare variables
  private RealType xcoord, ycoord, zcoord, vcoord;
  private RealTupleType domain_tuple;

  // The function (domain_tuple -> vcoord )
  private FunctionType func_domain_vcoord;

  // The Data values for the domain are represented by the Set
  private Set domain_set, intp_set;

  // The Data class FlatField
  private FlatField vals_ff;

  // The DataReference from data to display
  private DataReferenceImpl data_ref;

  // The value used for scaling data
  float Min_Xf, Max_Xf, Min_Yf, Max_Yf, Min_Zf, Max_Zf;

  /**
   * Interpolation3DLinear(double[] Xf, double[] Yf, double[] Zf, double[] Vf) is the
   * constructor of the class Interpolation3DLinear. It creates and instanciates
   * some particular variables required by VisAD to initialise the function v=F(x,y,z)
   * and the domain where its value is known. It also initialises numerically the
   * 3D-function to be interpolated. The vectors Xf, Yf and Zf define the definition
   * domain of the function F, and the vector Vf contains the values taken by F.
   */
  public Interpolation3DLinear(double[] Xf, double[] Yf, double[] Zf, double[] Vf)
  throws RemoteException, VisADException, NotSameLengthException
  {
    // Create the coordinates (x,y,z) system
    // Use VisAD RealType(String name);
    xcoord = InitInterpolation.xcoord;
    ycoord = InitInterpolation.ycoord;
    zcoord = InitInterpolation.zcoord;
    vcoord = InitInterpolation.vcoord;

    domain_tuple = new RealTupleType(xcoord , ycoord, zcoord);

    // Create a FunctionType (domain_tuple -> wcoord)
    // Use FunctionType(MathType domain, MathType range);
    func_domain_vcoord = new FunctionType( domain_tuple, vcoord);

    // Check that the length of all 4 arrays is the same
    int the_length1 = Xf.length;
    int the_length2 = Yf.length;
    int the_length3 = Zf.length;
    int the_length4 = Vf.length;
    if ((the_length1 != the_length2)||(the_length1 != the_length3)||(the_length1 != the_length4))
    {
      throw new NotSameLengthException("[dataFit][Interpolation3DLinear][constructor]: Arrays must have the same length.");
    }

    // Find out the Min and Max of Xf, Yf, Zf for
    // Converting data to the right scale later
    Min_Xf=(float)Xf[0];
    Min_Yf=(float)Yf[0];
    Max_Xf=(float)Xf[0];
    Max_Yf=(float)Yf[0];
    Min_Zf=(float)Zf[0];
    Max_Zf=(float)Zf[0];
    for (int i=0; i<the_length1; i++)
    {
      if (Xf[i]<=Min_Xf)
      {
        Min_Xf=(float)Xf[i];
      }
      if (Yf[i]<=Min_Yf)
      {
        Min_Yf=(float)Yf[i];
      }
      if (Xf[i]>=Max_Xf)
      {
        Max_Xf=(float)Xf[i];
      }
      if (Yf[i]>=Max_Yf)
      {
        Max_Yf=(float)Yf[i];
      }
      if (Zf[i]<=Min_Zf)
      {
        Min_Zf=(float)Zf[i];
      }
      if (Zf[i]>=Max_Zf)
      {
        Max_Zf=(float)Zf[i];
      }
    }

    // Create the domain Set
    // Conversion double-->float
    float[][] my_sample = new float[3][the_length1];
    for(int i=0; i < the_length1; i++)
    {
      my_sample[0][i]=((float)Xf[i]-Min_Xf)/(Max_Xf-Min_Xf);
      my_sample[1][i]=((float)Yf[i]-Min_Yf)/(Max_Yf-Min_Yf);
      my_sample[2][i]=((float)Zf[i]-Min_Zf)/(Max_Zf-Min_Zf);
    }
    domain_set = new Irregular3DSet(domain_tuple, my_sample);


    // For debugging only
    //for(int i=0; i< the_length1; i++)
    //{
      //System.out.println(i+"  x: "+my_sample[0][i]+"  y: "+my_sample[1][i]+"  z: "+my_sample[2][i]);
    //}

    // For debugging only
    //System.out.println("Xmin: "+Min_Xf);
    //System.out.println("Xmax: "+Max_Xf);
    //System.out.println("Ymin: "+Min_Yf);
    //System.out.println("Ymax: "+Max_Yf);
    //System.out.println("Zmin: "+Min_Zf);
    //System.out.println("Zmax: "+Max_Zf);

    // We create another array that will contains the values of vcoord
    float[][] flat_samples = new float[1][the_length1];
    for(int i=0; i < the_length1; i++)
    {
      flat_samples[0][i]=(float)Vf[i];
    }

    // Create a FlatField
    // Use FlatField(FunctionType type, Set domain_set)
    vals_ff = new FlatField( func_domain_vcoord, domain_set);

    // ...and put the values above into it
    // Note the argument false, meaning that the array won't be copied
    vals_ff.setSamples( flat_samples , false );

  }

  /** The method compute(double[] Xd, double[] Yd, double[] Zd) computes the
   * the interpolation by using a weighted average algorithm. When the
   * computing starts, the object on which the method is applied automatically
   * calls a Delaunay algorithm to generate the mesh that will support
   * the interpolation itself. (Xd,Yd,Zd) are the points where the function is
   * interpolated. This method is a wrapper of the VisAD method resample() and allows
   * interpolation at multiple points at a time. The compute method takes into account
   * the scalling factor proper.
   */
  public double[] compute(double[] Xd, double[] Yd, double[] Zd)
  throws RemoteException, VisADException, NotSameLengthException
  {
    // Checking that both arrays have same length
    int the_length5 = Xd.length;
    int the_length6 = Yd.length;
    int the_length7 = Zd.length;
    if ((the_length5 != the_length6)||(the_length5 != the_length7))
    {
      throw new NotSameLengthException("[dataFit][Interpolation3DLinear][compute]: Arrays must have the same length.");
    }

    // Conversion double-->float
    float[][] intp_sample = new float[3][the_length5];
    for(int i=0; i < the_length5; i++)
    {
      intp_sample[0][i]=((float)Xd[i]-Min_Xf)/(Max_Xf-Min_Xf);
      intp_sample[1][i]=((float)Yd[i]-Min_Yf)/(Max_Yf-Min_Yf);
      intp_sample[2][i]=((float)Zd[i]-Min_Zf)/(Max_Zf-Min_Zf);
    }
    intp_set = new Irregular3DSet(domain_tuple, intp_sample);

    // Compute the actual interpolation
    float[][] result = new float[1][the_length5];
    result=(vals_ff.resample(intp_set,DataImpl.WEIGHTED_AVERAGE,DataImpl.INDEPENDENT)).getFloats();

    // Return the results
    double[] final_result = new double[the_length5];
    for(int i=0; i<the_length5; i++)
    {
      final_result[i] = (double)result[0][i];
    }
    return final_result;
  }

  /**
   * The method evaluate(double Xd, double Yd, double Zd) computes the
   * the interpolation by using a weighted average algorithm. The data are
   * first scaled by (Xi-Min_X)/(Max_X-Min_X) (and the same for the array Y). When the
   * computing starts, the object on which the method is applied automatically
   * calls a Delaunay algorithm to generate the mesh that will support
   * the interpolation itself. (Xd,Yd,Zd) is the point where the function is
   * interpolated. This method is a wrapper of the VisAD method evaluate() and
   * only allows interpolation at a single point at a time.
   */
  public double evaluate (double Xd, double Yd, double Zd)
  throws RemoteException, VisADException, NotSameLengthException
  {
    Real xintp = new Real(xcoord, (Xd-(float)Min_Xf)/((float)Max_Xf-(float)Min_Xf));
    Real yintp = new Real(ycoord, (Yd-(float)Min_Yf)/((float)Max_Yf-(float)Min_Yf));
    Real zintp = new Real(zcoord, (Zd-(float)Min_Zf)/((float)Max_Zf-(float)Min_Zf));
    Real[] valint = new Real[] {xintp, yintp, zintp};
    RealTuple intp_tuple = new RealTuple(valint);
    Data result_temp = vals_ff.evaluate(intp_tuple,DataImpl.WEIGHTED_AVERAGE,DataImpl.INDEPENDENT);
    Real temp = (Real) result_temp;

    // For debugging only
    //System.out.println("Result : "+temp);

    double result = temp.getValue();
    return result;
  }

  /**
   * Main method. Only use for a demo of the Interpolation3DLinear
   */
  public static void main(String[] args)
  throws VisADException, RemoteException, NotSameLengthException
  {
      // Example of creation of an object of type Interpolation2DLinear
      double[] my_x = new double[] {-2, -1, 0, 1, 3, 0};
      double[] my_y = new double[] {0.5, 0, 1, 0, 1, -3};
      double[] my_z = new double[] {2, 0, -0.5, 1, 3, -1};
      double[] my_v = new double[] {0.62, 0.73, 0.58, 0.79, 0.34, 0.11};


      Interpolation3DLinear myInterpolation = new Interpolation3DLinear(my_x, my_y, my_z, my_v);
      Interpolation3DLinear myInterpolation2 = new Interpolation3DLinear(my_x,my_y,my_z,my_v);

      // Example of interpolation
      double[] my_xd = new double[] {0., -0.1};
      double[] my_yd = new double[] {0., 0.5};
      double[] my_zd = new double[] {0., 0.9};
      double[] result = new double[2];

      System.out.println("Now I interpolate at (0,0,0) and (-0.1,0.5,0.9) by using the method compute");
      result = myInterpolation2.compute(my_xd, my_yd, my_zd);
      System.out.println(result[0]);
      System.out.println(result[1]);

      // Example of interpolation with evaluate
      System.out.println("interpolation at (0,0,0) with method evaluate: "+myInterpolation.evaluate(0.,0.,0.));
      System.out.println("interpolation at (0,0,0) with method evaluate: "+myInterpolation2.evaluate(0,0,0));

    }
}